# Copyright (C) 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2018  Rickard Lindberg, Roger Lindberg
#
# This file is part of Timeline.
#
# Timeline is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Timeline is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Timeline.  If not, see <http://www.gnu.org/licenses/>.


from timelinelib.calendar.bosparanian.time import BosparanianTime
from timelinelib.test.cases.unit import UnitTestCase
from timelinelib.calendar.bosparanian.bosparaniandatetime import BosparanianDateTime
from timelinelib.calendar.bosparanian.bosparanian2julian import ymd_to_bosparanian_day
from timelinelib.calendar.bosparanian.julian2bosparanian import bosparanian_day_to_ymd


class DescribeBosparanianDateTime(UnitTestCase):

    def test_rejects_invalid_dates(self):
        """ """
        self.assertRaises(ValueError, BosparanianDateTime, 2013, 0, 1, 0, 0, 0)

    def test_can_replace(self):
        """ """
        g = BosparanianDateTime(2013, 7, 12, 10, 16, 12)
        self.assertEqual(g.replace(year=1990), BosparanianDateTime(1990, 7, 12, 10, 16, 12))
        self.assertEqual(g.replace(month=6), BosparanianDateTime(2013, 6, 12, 10, 16, 12))
        self.assertEqual(g.replace(year=1990, month=6), BosparanianDateTime(1990, 6, 12, 10, 16, 12))
        self.assertRaises(ValueError, g.replace, month=13)


class GregorianConversionsSpec(UnitTestCase):

    def test_can_convert_from_timeline_time_to_bosparanian(self):
        """ """
        self.assertEqual(
            BosparanianDateTime.from_time(BosparanianTime(julian_day=0, seconds=0)),
            BosparanianDateTime(-7300, 1, 4, 0, 0, 0))
        self.assertEqual(
            BosparanianDateTime.from_time(BosparanianTime(julian_day=1, seconds=0)),
            BosparanianDateTime(-7300, 1, 5, 0, 0, 0))

    def test_can_convert_from_bosparanian_to_timeline_time(self):
        """ """
        self.assertEqual(
            BosparanianDateTime(-7300, 1, 4, 0, 0, 0).to_time(),
            BosparanianTime(julian_day=0, seconds=0))
        self.assertEqual(
            BosparanianDateTime(-7300, 1, 5, 0, 0, 0).to_time(),
            BosparanianTime(julian_day=1, seconds=0))

    def test_roundtrip_julian_day_conversions(self):
        """ """
        for julian_day in range(100):
            (year, month, day) = bosparanian_day_to_ymd(julian_day)
            roundtrip = ymd_to_bosparanian_day(year, month, day)
            self.assertEqual(roundtrip, julian_day)

    def test_roundtrip_bosparanian_dates_conversions(self):
        """ """
        dates = [
            (2013, 1, 1),
            (2013, 1, 30),
        ]
        for bosparanian_date in dates:
            (year, month, day) = bosparanian_date
            julian_day = ymd_to_bosparanian_day(year, month, day)
            roundtrip = bosparanian_day_to_ymd(julian_day)
            self.assertEqual(roundtrip, bosparanian_date)


class DescribeBosparanianPrimitives(UnitTestCase):

    def test_is_valid(self):
        """ """
        self.assertTrue(BosparanianDateTime.is_valid(2013, 1, 1))
        self.assertFalse(BosparanianDateTime.is_valid(2013, 0, 1))
        self.assertTrue(BosparanianDateTime.is_valid(2013, 13, 1))
        self.assertFalse(BosparanianDateTime.is_valid(2013, 1, 0))
        self.assertFalse(BosparanianDateTime.is_valid(2013, 1, 32))
        self.assertTrue(BosparanianDateTime.is_valid(2013, 2, 30))

    def test_nbr_of_days_in_month(self):
        """ """
        self.assertEqual(30, BosparanianDateTime.nbr_of_days_in_month(2013, 1))
        self.assertEqual(30, BosparanianDateTime.nbr_of_days_in_month(2013, 2))
        self.assertEqual(30, BosparanianDateTime.nbr_of_days_in_month(2013, 3))
        self.assertEqual(30, BosparanianDateTime.nbr_of_days_in_month(2013, 4))
        self.assertEqual(30, BosparanianDateTime.nbr_of_days_in_month(2013, 5))
        self.assertEqual(30, BosparanianDateTime.nbr_of_days_in_month(2013, 6))
        self.assertEqual(30, BosparanianDateTime.nbr_of_days_in_month(2013, 7))
        self.assertEqual(30, BosparanianDateTime.nbr_of_days_in_month(2013, 8))
        self.assertEqual(30, BosparanianDateTime.nbr_of_days_in_month(2013, 9))
        self.assertEqual(30, BosparanianDateTime.nbr_of_days_in_month(2013, 10))
        self.assertEqual(30, BosparanianDateTime.nbr_of_days_in_month(2013, 11))
        self.assertEqual(30, BosparanianDateTime.nbr_of_days_in_month(2013, 12))
        self.assertEqual(30, BosparanianDateTime.nbr_of_days_in_month(2016, 2))
        self.assertEqual(5, BosparanianDateTime.nbr_of_days_in_month(2016, 13))

    def test_days_in_month(self):
        self.assertEqual(30, BosparanianDateTime.from_ymd(2013, 1, 1).days_in_month())
        self.assertEqual(30, BosparanianDateTime.from_ymd(2013, 2, 1).days_in_month())
        self.assertEqual(30, BosparanianDateTime.from_ymd(2016, 2, 1).days_in_month())
        self.assertEqual(30, BosparanianDateTime.from_ymd(2013, 3, 1).days_in_month())
        self.assertEqual(30, BosparanianDateTime.from_ymd(2013, 4, 1).days_in_month())
        self.assertEqual(30, BosparanianDateTime.from_ymd(2013, 5, 1).days_in_month())
        self.assertEqual(30, BosparanianDateTime.from_ymd(2013, 6, 1).days_in_month())
        self.assertEqual(30, BosparanianDateTime.from_ymd(2013, 7, 1).days_in_month())
        self.assertEqual(30, BosparanianDateTime.from_ymd(2013, 8, 1).days_in_month())
        self.assertEqual(30, BosparanianDateTime.from_ymd(2013, 9, 1).days_in_month())
        self.assertEqual(30, BosparanianDateTime.from_ymd(2013, 10, 1).days_in_month())
        self.assertEqual(30, BosparanianDateTime.from_ymd(2013, 11, 1).days_in_month())
        self.assertEqual(30, BosparanianDateTime.from_ymd(2013, 12, 1).days_in_month())
        self.assertEqual(5, BosparanianDateTime.from_ymd(2013, 13, 1).days_in_month())

    def test_year_is_leap_year(self):
        """ """
        self.assertFalse(BosparanianDateTime.year_is_leap_year(2013))
        self.assertFalse(BosparanianDateTime.year_is_leap_year(1900))
        self.assertFalse(BosparanianDateTime.year_is_leap_year(2016))
        self.assertFalse(BosparanianDateTime.year_is_leap_year(2000))

    def test_is_leap_year(self):
        """ """
        self.assertFalse(BosparanianDateTime.from_ymd(2013, 1, 1).is_leap_year)
        self.assertFalse(BosparanianDateTime.from_ymd(1900, 1, 1).is_leap_year)
        self.assertFalse(BosparanianDateTime.from_ymd(2016, 1, 1).is_leap_year)
        self.assertFalse(BosparanianDateTime.from_ymd(2000, 1, 1).is_leap_year)

    def test_days_in_year(self):
        self.assertEqual(365, BosparanianDateTime.from_ymd(2019, 1, 1).days_in_year)
        self.assertEqual(365, BosparanianDateTime.from_ymd(2020, 1, 1).days_in_year)

    def test_days_in_week(self):
        self.assertEqual(7, BosparanianDateTime.from_ymd(2019, 1, 1).days_in_week)

    def test_week_number(self):
        """ """
        def assert_is_week(date_tuple, w):
            (y, m, d) = date_tuple
            date = BosparanianDateTime.from_ymd(y, m, d)
            self.assertEqual(date.week_number, w)
        assert_is_week((2012, 12, 30), 52)
        assert_is_week((2012, 13, 1), 52)
        assert_is_week((2012, 13, 5), 52)
        assert_is_week((2013, 1, 1), 1)
        assert_is_week((2013, 1, 6), 1)
        assert_is_week((2013, 1, 11), 2)
        assert_is_week((2013, 7, 2), 26)

    def test_abbreviated_month_name(self):
        self.assertEqual(_('PRA'), BosparanianDateTime.from_ymd(2021, 1, 1).abbreviated_month_name)
        self.assertEqual(_('RON'), BosparanianDateTime.from_ymd(2021, 2, 1).abbreviated_month_name)
        self.assertEqual(_('EFF'), BosparanianDateTime.from_ymd(2021, 3, 1).abbreviated_month_name)
        self.assertEqual(_('TRA'), BosparanianDateTime.from_ymd(2021, 4, 1).abbreviated_month_name)
        self.assertEqual(_('BOR'), BosparanianDateTime.from_ymd(2021, 5, 1).abbreviated_month_name)
        self.assertEqual(_('HES'), BosparanianDateTime.from_ymd(2021, 6, 1).abbreviated_month_name)
        self.assertEqual(_('FIR'), BosparanianDateTime.from_ymd(2021, 7, 1).abbreviated_month_name)
        self.assertEqual(_('TSA'), BosparanianDateTime.from_ymd(2021, 8, 1).abbreviated_month_name)
        self.assertEqual(_('PHE'), BosparanianDateTime.from_ymd(2021, 9, 1).abbreviated_month_name)
        self.assertEqual(_('PER'), BosparanianDateTime.from_ymd(2021, 10, 1).abbreviated_month_name)
        self.assertEqual(_('ING'), BosparanianDateTime.from_ymd(2021, 11, 1).abbreviated_month_name)
        self.assertEqual(_('RAH'), BosparanianDateTime.from_ymd(2021, 12, 1).abbreviated_month_name)
        self.assertEqual(_('NL'), BosparanianDateTime.from_ymd(2021, 13, 1).abbreviated_month_name)

    def test_month_name(self):
        self.assertEqual(_('Praios'), BosparanianDateTime.from_ymd(2021, 1, 1).month_name)
        self.assertEqual(_('Rondra'), BosparanianDateTime.from_ymd(2021, 2, 1).month_name)
        self.assertEqual(_('Efferd'), BosparanianDateTime.from_ymd(2021, 3, 1).month_name)
        self.assertEqual(_('Travia'), BosparanianDateTime.from_ymd(2021, 4, 1).month_name)
        self.assertEqual(_('Boron'), BosparanianDateTime.from_ymd(2021, 5, 1).month_name)
        self.assertEqual(_('Hesinde'), BosparanianDateTime.from_ymd(2021, 6, 1).month_name)
        self.assertEqual(_('Firun'), BosparanianDateTime.from_ymd(2021, 7, 1).month_name)
        self.assertEqual(_('Tsa'), BosparanianDateTime.from_ymd(2021, 8, 1).month_name)
        self.assertEqual(_('Phex'), BosparanianDateTime.from_ymd(2021, 9, 1).month_name)
        self.assertEqual(_('Peraine'), BosparanianDateTime.from_ymd(2021, 10, 1).month_name)
        self.assertEqual(_('Ingerimm'), BosparanianDateTime.from_ymd(2021, 11, 1).month_name)
        self.assertEqual(_('Rahja'), BosparanianDateTime.from_ymd(2021, 12, 1).month_name)
        self.assertEqual(_('Nameless Days'), BosparanianDateTime.from_ymd(2021, 13, 1).month_name)

    def test_abbreviated_day_name(self):
        self.assertEqual(_('Earth'), BosparanianDateTime.from_ymd(2021, 1, 1).abbreviated_weekday_name)
        self.assertEqual(_('Market'), BosparanianDateTime.from_ymd(2021, 1, 2).abbreviated_weekday_name)
        self.assertEqual(_('Praios'), BosparanianDateTime.from_ymd(2021, 1, 3).abbreviated_weekday_name)
        self.assertEqual(_('Rohal'), BosparanianDateTime.from_ymd(2021, 1, 4).abbreviated_weekday_name)
        self.assertEqual(_('Fire'), BosparanianDateTime.from_ymd(2021, 1, 5).abbreviated_weekday_name)
        self.assertEqual(_('Water'), BosparanianDateTime.from_ymd(2021, 1, 6).abbreviated_weekday_name)
        self.assertEqual(_('Wind'), BosparanianDateTime.from_ymd(2021, 1, 7).abbreviated_weekday_name)
        self.assertEqual(_('Earth'), BosparanianDateTime.from_ymd(2021, 1, 8).abbreviated_weekday_name)

    def test_is_bc(self):
        self.assertFalse(BosparanianDateTime.from_ymd(2021, 1, 8).is_bc())
        self.assertFalse(BosparanianDateTime.from_ymd(-2021, 1, 8).is_bc())

    def test_formatted_year(self):
        self.assertEqual('2021BF', BosparanianDateTime.from_ymd(2021, 1, 8).formatted_year)
        self.assertEqual('-2021BF', BosparanianDateTime.from_ymd(-2021, 1, 8).formatted_year)

    def test_last_month(self):
        self.assertEqual(13, BosparanianDateTime.from_ymd(2021, 1, 8).last_month)
        self.assertEqual(13, BosparanianDateTime.from_ymd(-2021, 1, 8).last_month)
