# Copyright (C) 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2018  Rickard Lindberg, Roger Lindberg
#
# This file is part of Timeline.
#
# Timeline is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Timeline is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Timeline.  If not, see <http://www.gnu.org/licenses/>.


import timelinelib.wxgui.frames.mainframe.menus as mid


class MenuController:
    """
    The MenuController is "owned" by the
    :class:`~timelinelib.wxgui.frames.mainframe.mainframe.MainFrame` class.

    It is used for functions that enables/disables menu items which is done
    with the function update_menus_enabled_state().
    """

    def __init__(self, menu_bar):
        self._timeline = None
        self._menu_bar = menu_bar
        self.menus_requiring_timeline = []
        self.menus_requiring_writable_timeline = []
        self.menus_requiring_visible_timeline_view = []

    def update_menus_enabled_state(self, timeline_view_visible, nbr_of_selected_events):
        """
        This function is called whenever a new timeline is opened or when the state of the
        current timeline is changed.

        It enables or disables menus depending on the state of the timeline.
        """
        some_event_selected = nbr_of_selected_events > 0
        one_event_selected = nbr_of_selected_events == 1
        two_events_selected = nbr_of_selected_events == 2
        self._enable_disable_menus(timeline_view_visible)
        self._enable(mid.ID_MOVE_EVENT_UP, one_event_selected)
        self._enable(mid.ID_MOVE_EVENT_DOWN, one_event_selected)
        self._enable(mid.ID_MOVE_EVENTS, some_event_selected)
        self._enable(mid.ID_SET_CATEGORY_ON_SELECTED, some_event_selected)
        self._enable(mid.ID_DUPLICATE_EVENT, one_event_selected)
        self._enable(mid.ID_EDIT_EVENT, one_event_selected)
        self._enable(mid.ID_MEASURE_DISTANCE, two_events_selected)
        self._enable(mid.ID_UNDO, self._timeline.undo_enabled() if self._has_timeline else False)
        self._enable(mid.ID_REDO, self._timeline.redo_enabled() if self._has_timeline else False)

    def on_timeline_change(self, timeline):
        """
        Called when a new timeline is opened.

        It stores the timeline to later be able to figure out it's state.
        """
        self._timeline = timeline

    def add_menu_requiring_writable_timeline(self, menu):
        """Add a menu to the list of menus that requires a writeable timeline."""
        self.menus_requiring_writable_timeline.append(menu)

    def add_menu_requiring_timeline(self, menu):
        """Add a menu to the list of menus that requires a timeline."""
        self.menus_requiring_timeline.append(menu)

    def add_menu_requiring_visible_timeline_view(self, menu):
        """Add a menu to the list of menus that requires a visible timeline."""
        self.menus_requiring_visible_timeline_view.append(menu)

    @property
    def _has_timeline(self):
        return self._timeline is not None

    @property
    def _timeline_is_readonly(self):
        return self._has_timeline and self._timeline.is_read_only()

    def _enable_disable_menus(self, timeline_view_visible):
        for menu in self.menus_requiring_writable_timeline:
            self._enable_disable_menu_requiring_writable_timeline(menu)
        for menu in self.menus_requiring_timeline:
            self._enable_disable_menu_requiring_timeline(menu)
        for menu in self.menus_requiring_visible_timeline_view:
            self._enable_disable_menu_requiring_visible_timeline_view(menu, timeline_view_visible)

    def _enable_disable_menu_requiring_writable_timeline(self, menu):
        if not self._has_timeline or self._timeline_is_readonly:
            menu.Enable(False)
        else:
            menu.Enable(True)

    def _enable_disable_menu_requiring_timeline(self, menu):
        menu.Enable(self._has_timeline)

    def _enable_disable_menu_requiring_visible_timeline_view(self, menu, timeline_view_visible):
        menu.Enable(self._has_timeline  and timeline_view_visible)

    def _enable(self, menu_id, value):
        mnu = self._menu_bar.FindItemById(menu_id)
        mnu.Enable(value)
