# Copyright (C) 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2018  Rickard Lindberg, Roger Lindberg
#
# This file is part of Timeline.
#
# Timeline is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Timeline is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Timeline.  If not, see <http://www.gnu.org/licenses/>.
from timelinelib.canvas.data import Subevent
from timelinelib.repositories.dbwrapper import DbWrapperEventRepository
from timelinelib.wxgui.framework import Controller


class SetContainerDialogController(Controller):

    def on_init(self, parent, db, selected_event_ids):
        self._parent = parent
        self._db = db
        self._selected_event_ids = selected_event_ids
        self.view.PopulateContainers()
        self.view.SetTitle(_("Set Container on selected events"))

    def on_ok_clicked(self, event=None):
        container = self.view.GetSelectedContainer()
        if self._is_new_container(container):
            self._add_new_container(container)
        if not self._container_is_given(container) and self._selected_event_ids == []:
            self.view.DisplayErrorMessage(_("You must select a container!"))
        else:
            self._save_container_in_events(container)
            self.view.EndModalOk()

    def _is_new_container(self, container):
        return container not in self._db.get_containers()

    def _add_new_container(self, container):
        DbWrapperEventRepository(self._db).save(container)

    def _container_is_given(self, container):
        return container is not None

    def _save_container_in_events(self, container):
        with self._db.transaction("Set container"):
            for event_id in self._selected_event_ids:
                event = self._db.find_event_with_id(event_id)
                if event:
                    subevent = self._create_subevent(event, container)
                    subevent.set_container(container)
                    subevent.save()

    def _create_subevent(self, event, container):
        subevent = Subevent(db=self._db).update(
            event.get_time_period().start_time,
            event.get_time_period().end_time,
            event.text,
            event.category,
            ends_today=event.get_ends_today(),
            fuzzy_start=event.fuzzy_start,
            fuzzy_end=event.fuzzy_end,
            fuzzy=event.get_fuzzy(),
            locked=event.get_locked()
        )
        subevent.container = container
        subevent.data = event.data
        self._parent.view_properties.set_selected(event, is_selected=False)
        self._db.delete_event(event)
        return subevent
