# Copyright (C) 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2018  Rickard Lindberg, Roger Lindberg
#
# This file is part of Timeline.
#
# Timeline is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Timeline is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Timeline.  If not, see <http://www.gnu.org/licenses/>.


from timelinelib.wxgui.dialogs.eventsmove.controller import EventsMoveController
from timelinelib.wxgui.framework import Dialog
from timelinelib.db.utils import safe_locking


class EventsMoveDialog(Dialog):

    """
    <BoxSizerVertical>
        <BoxSizerVertical border="LEFT">
            <StaticText
                label=""
            />
            <StaticText
                label="$(explanation_text1)"
            />
            <StaticText
                label="$(explanation_text2)"
            />
            <StretchSpacer />
        </BoxSizerVertical>
        <FlexGridSizer
            rows="12"
            columns="2"
            growableColumns="1"
            proportion="1"
            border="ALL">

            <StaticText
                align="ALIGN_CENTER_VERTICAL"
                label="$(category_text)"
            />
            <CategoryChoice
                name="category_choice"
                allow_add="False"
                timeline="$(db)"
                align="ALIGN_CENTER_VERTICAL"
            />
            <StaticText
                align="ALIGN_CENTER_VERTICAL"
                label="$(period_start_text)"
            />
            <BoxSizerHorizontal>
                <TimePicker
                    name="start_time"
                    show_time="False"
                    time_type="$(time_type)"
                    config="$(config)"
                />
                <StretchSpacer size="100"/>
            </BoxSizerHorizontal>
            <StaticText
                align="ALIGN_CENTER_VERTICAL"
                label="$(period_end_text)"
            />
            <BoxSizerHorizontal>
                <TimePicker
                    name="end_time"
                    time_type="$(time_type)"
                    config="$(config)"
                    show_time="False"
                />
            </BoxSizerHorizontal>
            <StretchSpacer />
            <StretchSpacer />
            <StretchSpacer />
            <StretchSpacer />
            <StretchSpacer />
            <StretchSpacer />
        </FlexGridSizer>

        <DialogButtonsOkCancelSizer
            border="LEFT|BOTTOM|RIGHT"
            event_EVT_BUTTON__ID_OK="on_ok_clicked"
        />

    </BoxSizerVertical>
    """

    ALL_CATEGORIES = _('All Categories')

    def __init__(self, parent, title, db, config, preferred_category):
        Dialog.__init__(self, EventsMoveController, parent, {
            "db": db,
            "category_text": _("Category:"),
            "period_start_text": _("From sample date:"),
            "period_end_text": _("Move to date:"),
            "explanation_text1": _("Give two dates to define the"),
            "explanation_text2": _("number of days to move events."),
            "time_type": db.get_time_type(),
            "config": config,
            "result_size": (-1, 50),
        }, title=title, size=(100, -1))
        self.controller.on_init(parent, db, config, preferred_category)

    def PopulateCategories(self, exclude):
        self.category_choice.Populate(exclude=exclude)
        self.category_choice.Delete(0)  # Remove blank line
        self.category_choice.Insert(self.ALL_CATEGORIES, 0, None)
        self.Fit()

    def SelectCategory(self, inx):
        return self.category_choice.Select(inx)

    def SetPreferredCategory(self, preferred_category_name):
        for inx in range(self.category_choice.GetCount()):
            category_name = self.category_choice.GetString(inx).strip()
            if category_name == preferred_category_name:
                self.category_choice.SetSelection(inx)
                break

    def GetCategory(self):
        return self.category_choice.GetSelectedCategory()

    def SetStartTime(self, value):
        self.start_time.set_value(value)

    def GetStartTime(self):
        if self.start_time.IsEnabled():
            return self.start_time.get_value()

    def SetEndTime(self, value):
        self.end_time.set_value(value)

    def GetEndTime(self):
        if self.end_time.IsEnabled():
            return self.end_time.get_value()


def open_events_move_dialog(main_frame, preferred_category=EventsMoveDialog.ALL_CATEGORIES):

    def create_editor():
        return EventsMoveDialog(main_frame, _('Move selected Events in time'),
                              main_frame.timeline, main_frame.config, preferred_category)

    def edit_function():
        dialog = create_editor()
        dialog.ShowModal()
        dialog.Destroy()

    safe_locking(main_frame.controller, edit_function)
