# Copyright (C) 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2018  Rickard Lindberg, Roger Lindberg
#
# This file is part of Timeline.
#
# Timeline is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Timeline is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Timeline.  If not, see <http://www.gnu.org/licenses/>.


from timelinelib.general.observer import Listener
from timelinelib.wxgui.components.timelinepanelguicreator import TimelinePanelGuiCreator


class TimelinePanel(TimelinePanelGuiCreator):
    """
    This panel is used to render a timeline.

    The panel contains:
        * splitter window (containing sidebar and main canvas)
        * toolbar
        * divider line slider

    Events handled by this window:

        wx.EVT_SLIDER
            Triggered when the divider line slider position is changed.

        wx.EVT_CONTEXT_MENU
            Triggered when the right mouse button is used in the divider line slider.

        wx.EVT_SPLITTER_SASH_POS_CHANGED
            Triggered when the width of the sidebar is changed.

        wx.EVT_LEFT_DCLICK
            Triggered when main canvas is doubled-clicked with the left mouse button.

        wx.EVT_RIGHT_DOWN
            Triggered when the right mouse button is pressed in the main canvas.

        wx.EVT_KEY_DOWN
            Triggered when a keyboard key is pressed in the main canvas.

        EVT_DIVIDER_POSITION_CHANGED
            Triggered when the divider line slider position is changed.

        EVT_TIMELINE_REDRAWN
            Triggered when the timeline is redrawn.

    """

    def __init__(self, parent, main_frame):
        self._main_frame = main_frame
        self._config = main_frame.config
        TimelinePanelGuiCreator.__init__(self, parent)
        self._db_listener = Listener(self._on_db_changed)
        self._config.listen_for('show_sidebar', self._on_show_sidebar_changed)
        self._config.listen_for('show_legend', self._on_show_legend_changed)
        self._config.listen_for('show_label_filtering', self._on_show_label_filtering)

    @property
    def sidebar_width(self):
        return self._sidebar_width
    
    def Activated(self):
        """
        This function is called whenever the timeline panel is shown.

        Show or hide the sidebar depending on config settings.
        """
        if self._config.show_sidebar:
            self._show_sidebar()

    def SetTimeline(self, timeline):
        """Register a new timeline."""
        self.timeline_canvas.SetTimeline(timeline)
        self._db_listener.set_observable(timeline)

    def _show_sidebar(self):
        self.splitter.SplitVertically(self.sidebar, self.timeline_canvas, self._sidebar_width)
        self.splitter.SetSashPosition(self._sidebar_width)
        self.splitter.SetMinimumPaneSize(self.sidebar.GetBestSize()[0])

    def _on_db_changed(self, db):
        if db.is_read_only():
            header = _("This timeline is read-only.")
            body = _("To edit this timeline, save it to a new file: File -> Save As.")
            self.message_bar.ShowInformationMessage("%s\n%s" % (header, body))
        elif not db.is_saved():
            header = _("This timeline is not being saved.")
            body = _("To save this timeline, save it to a new file: File -> Save As.")
            self.message_bar.ShowWarningMessage("%s\n%s" % (header, body))
        else:
            self.message_bar.ShowNoMessage()

    def _on_show_sidebar_changed(self):
        if self._config.show_sidebar:
            self._show_sidebar()
        else:
            self._hide_sidebar()

    def _on_show_legend_changed(self):
        self._main_frame.canvas.Redraw()

    def _hide_sidebar(self):
        self.splitter.Unsplit(self.sidebar)

    def _on_show_label_filtering(self):
        if self._config.show_label_filtering:
            self._show_label_filtering()
        else:
            self._hide_label_filtering()

    def _show_label_filtering(self):
        self.sidebar.show_label_filtering()

    def _hide_label_filtering(self):
        self.sidebar.hide_label_filtering()

