# Copyright (C) 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2018  Rickard Lindberg, Roger Lindberg
#
# This file is part of Timeline.
#
# Timeline is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Timeline is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Timeline.  If not, see <http://www.gnu.org/licenses/>.


import wx

from timelinelib.wxgui.components.welcomepanel import WelcomePanel
from timelinelib.wxgui.components.timelinepanel import TimelinePanel
from timelinelib.wxgui.components.searchbar.view import SearchBar
from timelinelib.wxgui.components.filedroptarget import FileDropTarget


class MainPanel(wx.Panel):
    """
    Panel that covers the whole client area of MainFrame.

    Displays one of the following panels:

      * The :class:`~timelinelib.wxgui.components.welcomepanel.WelcomePanel`
      * The :class:`~timelinelib.wxgui.components.timelinepanel.TimelinePanel`

    It also handles the :class:`~timelinelib.wxgui.components.searchbar.view.SearchBar`
    """

    def __init__(self, main_frame):
        wx.Panel.__init__(self, main_frame)
        self._welcome_panel = None
        self._timeline_panel = None
        self._searchbar = None
        self._prev_time_period = None
        self._main_frame = main_frame
        self._create_gui()

    @property
    def timeline_panel(self):
        return self._timeline_panel

    def DisplayWelcomePanel(self):
        self._show_panel(self._welcome_panel)

    def DisplayTimelinePanel(self, timeline):
        if timeline is None:
            # Do we ever end up here with the welcome panel displayed?
            self._remove_timeline_and_show_welcome_panel()
        else:
            self._show_new_timeline(timeline)

    def DisplaySearchbar(self, show=True):
        """Show or hide the searchbar."""
        self._searchbar.Show(show)
        if show is True:
            self._searchbar.Focus()
        self.GetSizer().Layout()

    def TimelinePanelVisible(self):
        """Return True if the TimelinePanel is shown."""
        return self._timeline_panel.IsShown()

    def _create_gui(self):
        # Search bar
        self._searchbar = SearchBar(self)
        self._searchbar.Show(False)
        # Panels
        self._welcome_panel = WelcomePanel(self, self._main_frame)
        self._timeline_panel = TimelinePanel(self, self._main_frame)
        # Layout
        self.sizerOuter = wx.BoxSizer(wx.VERTICAL)
        self.sizer = wx.BoxSizer(wx.HORIZONTAL)
        self.sizer.Add(self._welcome_panel, flag=wx.GROW, proportion=1)
        self.sizer.Add(self._timeline_panel, flag=wx.GROW, proportion=1)
        self.sizerOuter.Add(self.sizer, flag=wx.GROW, proportion=1)
        self.sizerOuter.Add(self._searchbar, flag=wx.GROW, proportion=0)
        self.SetSizer(self.sizerOuter)

    def _show_panel(self, panel):
        self._hide_all_panels()
        panel.Show(True)
        self.sizerOuter.Layout()
        panel.Activated()

    def _hide_all_panels(self):
        for panel_to_hide in [self._welcome_panel, self._timeline_panel]:
            panel_to_hide.Show(False)

    def _show__timeline_panel(self):
        self._show_panel(self._timeline_panel)

    def _remove_timeline_and_show_welcome_panel(self):
        self._timeline_panel.sidebar.category_tree.set_no_timeline_view()
        self._set_searchbar_timeline_canvas(None)
        self._timeline_panel.SetTimeline(None)
        self.DisplayWelcomePanel()

    def _show_new_timeline(self, timeline):
        self._prev_time_period = None
        self._timeline_panel.SetTimeline(timeline)
        canvas = self._main_frame.canvas
        self._timeline_panel.sidebar.category_tree.set_timeline_view(canvas.GetDb(), canvas.GetViewProperties())
        self._set_searchbar_timeline_canvas(canvas)
        self._show__timeline_panel()
        canvas.SetDropTarget(FileDropTarget(canvas))

    def _set_searchbar_timeline_canvas(self, timeline_canvas):
        self._searchbar.SetTimelineCanvas(timeline_canvas)
