# Copyright (C) 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2018  Rickard Lindberg, Roger Lindberg
#
# This file is part of Timeline.
#
# Timeline is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Timeline is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Timeline.  If not, see <http://www.gnu.org/licenses/>.

from timelinelib.config.shortcut import NO_MODIFIER
DOTS = "\u2026"     # Unicode character for ...


class MenuItemInfo:
    """
    This class holds information about a menu item.

    MenuItemInfo objects are created in the menu class to which they belong.
    (:class:`~timelinelib.wxgui.frames.mainframe.menus.filemenu.FileMenu`,
    :class:`~timelinelib.wxgui.frames.mainframe.menus.editmenu.EditMenu`,
    :class:`~timelinelib.wxgui.frames.mainframe.menus.viewmenu.ViewMenu`,
    :class:`~timelinelib.wxgui.frames.mainframe.menus.timelinemenu.TimelineMenu`,
    :class:`~timelinelib.wxgui.frames.mainframe.menus.navigatemenu.NavigateMenu`,
    :class:`~timelinelib.wxgui.frames.mainframe.menus.helpmenu.HelpMenu`)

    :doc:`Tests are found here <unit_config_menuiteminfo>`.
    """

    def __init__(self, wxid=0, menu_item=None, config_key="", config_display_name="",
                 shortcut_key="", shortcut_modifier=NO_MODIFIER, event_handler=None,
                 sort_order=0):
        self._wxid = wxid
        self._menu_item = menu_item
        self._config_key = config_key
        self._config_display_name = config_display_name.replace(DOTS, "").replace("...", "").replace('&', '')
        self._shortcut_key = shortcut_key
        self._shortcut_modifiers = shortcut_modifier
        self._event_handler = event_handler
        self._sort_order = sort_order

    @property
    def wxid(self):
        """The ID associated with a menu item."""
        return self._wxid

    @property
    def menu_item(self):
        """The menu item associated with the wxid."""
        return self._menu_item

    @property
    def config_key(self):
        """
        The key in the timeline configuration file where the shortcut
        information for the menu item is stored.
        """
        return self._config_key

    @menu_item.setter
    def menu_item(self, value):
        self._menu_item = value

    @property
    def config_display_name(self):
        """
        The name displayed in the
        :class:`~timelinelib.wxgui.dialogs.shortcutseditor.view.ShortcutsEditorDialog`
        dialog to identify the menu item.
        """
        return self._config_display_name

    @config_display_name.setter
    def config_display_name(self, value):
        self._config_display_name = value.replace('...', '').replace('&', '')

    @property
    def shortcut_key(self):
        """The menu item shortcut key."""
        return self._shortcut_key

    @shortcut_key.setter
    def shortcut_key(self, value):
        """The menu item shortcut key string."""
        self._shortcut_key = value

    @property
    def shortcut_modifiers(self):
        """The menu item shortcut modifiers string."""
        return self._shortcut_modifiers

    @shortcut_modifiers.setter
    def shortcut_modifiers(self, value):
        self._shortcut_modifiers = value

    @property
    def shortcut(self):
        """The complete shortcut string used for the menu item."""
        return f'{self._shortcut_modifiers}+{self._shortcut_key}'

    @property
    def event_handler(self):
        """The function that handles the menu event when the menu item is selected."""
        return self._event_handler

    @event_handler.setter
    def event_handler(self, value):
        self._event_handler = value

    @property
    def sort_order(self):
        """
        An integer describing the sort order of the menu item within
        the menu to which the menu item belongs.
        """
        return self._sort_order
