# Copyright (C) 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2018  Rickard Lindberg, Roger Lindberg
#
# This file is part of Timeline.
#
# Timeline is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Timeline is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Timeline.  If not, see <http://www.gnu.org/licenses/>.

"""
The ApplicationArguments class parses the command line arguments and
options when the application is started.

If the application is started with:
    python3 timeline.py -h
    
a text will be displayed that describes valid arguments and valid
options.

:doc:`Tests are found here <unit_config_arguments>`.
"""


from optparse import OptionParser
import os.path
import wx

from timelinelib.meta.version import get_full_version


FILENAME_DOC = """\
Arguments:

   filename:  If given, should be the path to a timeline file or one of the following special values:
        * :tutorial:
            Creates a readonly Gregorian tutorial timeline 
        * :numtutorial:                             
            Creates a readonly Numeric tutorial timeline   
        * path to an .ics file                      
            Creates a readonly timeline from ics data
        * path to a directory                          
            Creates a readonly timeline with the directory content as events
"""


class PosOptionParser(OptionParser):
    """Extend OptionParser with the possibility to print a formatted epilog in the help output."""

    def __init__(self, *args, **kwargs):
        """Init parent class"""
        OptionParser.__init__(self, *args, **kwargs)

    def format_help(self, formatter=None):
        """Extend the functionallity of the format_help() function."""
        return super().format_help(formatter) + f'\n\n{FILENAME_DOC}'


class ApplicationArguments:
    """
    Parse and store the application arguments and options given on the command line when
    Timeline is started.
    """

    def __init__(self, arguments):
        """Create the command line parser and parse the arguments and options."""
        self._options = None
        self._arguments = None
        self._option_parser = self._create_option_parser()
        self._create_config_file_option()
        self._create_debug_option()
        self._parse_from(arguments)

    @property
    def config_file_path(self):
        """
        Return the path to the timeline configuration file.

        If the path is not given on the commandline the following path is returned:
            wx.StandardPaths.Get().GetUserConfigDir() + ".thetimelineproj.cfg"
        """
        if self._options.config_file_path:
            return self._options.config_file_path
        else:
            return os.path.join(
                wx.StandardPaths.Get().GetUserConfigDir(),
                ".thetimelineproj.cfg")

    @property
    def first_file(self):
        """
        Return the filename argument given on the command line
        or None if no filename was given.
        """
        try:
            return self._arguments[0]
        except IndexError:
            return None

    @property
    def debug_flag(self):
        """Return True/False depending on the ``--debug`` flag is used or not."""
        return self._options.debug

    def has_files(self):
        """Return True/False depending on the existance of the filename argument"""
        return len(self._arguments) > 0

    def _get_files(self):
        return self._arguments

    def _create_option_parser(self):
        return PosOptionParser(
            usage="%prog [options] [filename]",
            version=self._create_version_string())
        
    def _create_version_string(self):
        return "%prog " + get_full_version()
    
    def _create_config_file_option(self):
        self._option_parser.add_option(
            "-c", "--config-file", dest="config_file_path", default=None,
            help="Path to config file")

    def _create_debug_option(self):
        self._option_parser.add_option(
            "--debug",
            default=False, action="store_true",
            help="Run Timeline with extra debug output")

    def _parse_from(self, arguments):
        (self._options, self._arguments) = self._option_parser.parse_args(arguments)
