# Copyright (C) 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2018  Rickard Lindberg, Roger Lindberg
#
# This file is part of Timeline.
#
# Timeline is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Timeline is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Timeline.  If not, see <http://www.gnu.org/licenses/>.


import os

import wx

from timelinelib.canvas.drawing.utils import black_solid_pen, black_solid_brush, get_colour, darken_color
from timelinelib.config.paths import EVENT_ICONS_DIR
from timelinelib.canvas.eventboxdrawers.defaultmilestonedrawer import DefaultMilestoneDrawer
from timelinelib.canvas.eventboxdrawers.handlerect import HandleRect, MIDDLE_HANDLE, LEFT_HANDLE, RIGHT_HANDLE
from timelinelib.wxgui.utils import load_bitmap
from timelinelib.canvas.drawing.rect import Rect


HANDLE_SIZE = 4
HALF_HANDLE_SIZE = HANDLE_SIZE // 2
DATA_INDICATOR_SIZE = 10
GRAY = (200, 200, 200)
BITMAPS = dict();


class DefaultEventBoxDrawer:

    def draw(self, dc, scene, event_rects, view_properties):
        rect = event_rects.box_rect
        event = event_rects.event
        self.scene = scene
        self.view_properties = view_properties
        selected = view_properties.is_selected(event)
        self.center_text = scene.center_text()
        if event.is_milestone():
            DefaultMilestoneDrawer(rect, event, selected, view_properties).draw(dc)
        elif scene.draw_short_period_event_as_symbol_below_divider_line() and rect.is_allowed_to_overlap() and event.is_period():
            self._draw_period_event_as_symbol_below_divider_line(dc, rect, scene, event)
        else:
            self._draw_event_box(dc, event_rects, selected)

    def _draw_period_event_as_symbol_below_divider_line(self, dc, rect, scene, event):
        x = rect.X + rect.Width // 2
        y0 = scene.divider_y
        y1 = rect.Y + rect.Height // 2
        dc.SetBrush(black_solid_brush())
        dc.SetPen(black_solid_pen(1))
        dc.DrawLine(x, y0, x, y1)
        dc.DrawCircle(x, y1, 2)

    def _draw_event_box(self, dc, event_rects, selected):
        event = event_rects.event
        rect = event_rects.box_rect
        self._draw_background(dc, rect, event)
        self._draw_fuzzy_edges(dc, rect, event)
        self._draw_locked_edges(dc, rect, event)
        self._draw_progress_box(dc, rect, event)
        self._draw_text(dc, event_rects)
        self._draw_contents_indicator(dc, event, rect)
        self._draw_locked_edges(dc, rect, event)
        self._draw_selection_handles(dc, event, rect, selected)
        self._draw_hyperlink(dc, rect, event)

    def _draw_background(self, dc, rect, event):
        dc.SetBrush(wx.Brush(event.get_color(), wx.BRUSHSTYLE_SOLID))
        dc.SetPen(self._get_pen(dc, event))
        dc.DrawRectangle(rect)

    def _get_pen(self, dc, event):
        pen = self._get_thin_border_pen(event)
        if self.view_properties.is_highlighted(event):
            if self.view_properties.get_highlight_count(event) % 2 == 0:
                pen = self._get_thick_border_pen(event)
        return pen

    def _draw_fuzzy_edges(self, dc, rect, event):
        if event.fuzzy_start:
            self._draw_fuzzy_start(dc, rect)
        if event.fuzzy_end:
            if not event.get_ends_today():
                self._draw_fuzzy_end(dc, rect)

    def _draw_locked_edges(self, dc, rect, event):
        if event.get_ends_today():
            self._draw_locked_end(dc, rect)
        if event.get_locked():
            self._draw_locked_start(dc, rect)
            self._draw_locked_end(dc, rect)

    def _draw_contents_indicator(self, dc, event, rect):
        if event.has_balloon_data():
            self._draw_balloon_indicator(dc, event, rect)

    def _draw_selection_handles(self, dc, event, rect, selected):
        if not event.locked and selected:
            self._draw_handles(dc, event, rect)

    def _get_thin_border_pen(self, event):
        return self._get_border_pen(event)

    def _get_thick_border_pen(self, event):
        return self._get_border_pen(event, thickness=8)

    def _get_border_pen(self, event, thickness=1):
        return wx.Pen(event.get_border_color(), thickness, wx.PENSTYLE_SOLID)

    def _get_balloon_indicator_brush(self, event):
        base_color = event.get_color()
        darker_color = darken_color(base_color, 0.6)
        brush = wx.Brush(darker_color, wx.BRUSHSTYLE_SOLID)
        return brush

    def _draw_fuzzy_start(self, dc, rect):
        dc.DrawBitmap(self._get_fuzzy_bitmap(), rect.x - 4, rect.y + 4, True)

    def _draw_fuzzy_end(self, dc, rect):
        dc.DrawBitmap(self._get_fuzzy_bitmap(), rect.x + rect.width - 8, rect.y + 4, True)

    def _draw_locked_start(self, dc, rect):
        dc.DrawBitmap(self._get_lock_bitmap(), rect.x - 7, rect.y + 3, True)

    def _draw_locked_end(self, dc, rect):
        dc.DrawBitmap(self._get_lock_bitmap(), rect.x + rect.width - 8, rect.y + 3, True)

    def _draw_progress_box(self, dc, rect, event):
        if event.get_data("progress"):
            self._set_progress_color(dc, event)
            progress_rect = self._get_progress_rect(rect, event)
            dc.DrawRectangle(progress_rect)

    def _set_progress_color(self, dc, event):
        progress_color = event.get_progress_color()
        dc.SetBrush(wx.Brush(wx.Colour(progress_color[0], progress_color[1], progress_color[2])))

    def _get_progress_rect(self, event_rect, event):
        HEIGHT_FACTOR = 0.35
        h = event_rect.height * HEIGHT_FACTOR
        y = event_rect.y + (event_rect.height - h)
        if event.is_period():
            rw = self.scene._calc_width_for_period_event(event)
            rx = self.scene._calc_x_pos_for_period_event(event)
        else:
            rw = self.scene._calc_width_for_non_period_event(event)
            rx = self.scene._calc_x_pos_for_non_period_event(event, rw)
        w = rw * event.get_data("progress") / 100.0
        # Avoid overflow values in args to wx.Rect()
        #  which can happen when rx has a very large negative or positive value.
        if rx < 0:
            w += rx
            rx = 0
        if w > self.scene.width:
            w = self.scene.width
        return Rect(int(rx), int(y), int(w), int(h))

    def _draw_balloon_indicator(self, dc, event, rect):
        """
        The data contents indicator is a small triangle drawn in the upper
        right corner of the event rectangle.
        """
        corner_x = rect.X + rect.Width
        points = (
            wx.Point(corner_x - DATA_INDICATOR_SIZE, rect.Y),
            wx.Point(corner_x, rect.Y),
            wx.Point(corner_x, rect.Y + DATA_INDICATOR_SIZE),
        )
        dc.SetBrush(self._get_balloon_indicator_brush(event))
        dc.SetPen(wx.TRANSPARENT_PEN)
        dc.SetClippingRegion(rect)
        dc.DrawPolygon(points)
        dc.DestroyClippingRegion()

    def _draw_text(self, dc, event_rects):
        # Ensure that we can't draw content outside inner rectangle
        if self._there_is_room_for_the_text(event_rects.text_rect):
            self._draw_the_text(dc, event_rects)

    def _there_is_room_for_the_text(self, rect):
        return rect.Width > 0

    def _draw_the_text(self, dc, event_rects):
        self._set_text_foreground_color(dc, event_rects)
        self._draw_normal_text(dc, event_rects.text_rect, event_rects.event)

    def _draw_normal_text(self, dc, rect, event):
        dc.SetClippingRegion(rect)
        dc.DrawText(self.scene.get_event_text(event), rect.X, rect.Y)
        dc.DestroyClippingRegion()

    def _set_text_foreground_color(self, dc, event_rects):
        if event_rects.text_outside_box():
            color = wx.BLACK
        else:
            try:
                color = get_colour(event_rects.event.get_category().font_color)
            except AttributeError:
                color = wx.BLACK
        dc.SetTextForeground(color)

    def _draw_handles(self, dc, event, rect):

        def draw_frame_around_event():
            small_rect = rect.CloneDeflate(1, 1)
            border_color = event.get_border_color()
            border_color = darken_color(border_color)
            pen = wx.Pen(border_color, 1, wx.PENSTYLE_SOLID)
            dc.SetBrush(wx.TRANSPARENT_BRUSH)
            dc.SetPen(pen)
            dc.DrawRectangle(small_rect)

        draw_frame_around_event()
        self._draw_all_handles(dc, rect)

    @staticmethod
    def _draw_all_handles(dc, rect):
        HandleRect(rect, LEFT_HANDLE).draw(dc)
        HandleRect(rect, MIDDLE_HANDLE).draw(dc)
        HandleRect(rect, RIGHT_HANDLE).draw(dc)

    def _draw_hyperlink(self, dc, rect, event):
        if event.get_hyperlink():
            bitmap = self._get_hyperlink_bitmap()
            if not bitmap.IsOk():
                bitmap = wx.ArtProvider.GetBitmap(wx.ART_MISSING_IMAGE, wx.ART_MENU)
            dc.DrawBitmap(bitmap, rect.x + rect.width - 14, rect.y + 4, True)

    def _get_hyperlink_bitmap(self):
        return get_bitmap(self.view_properties.get_hyperlink_icon())

    def _get_lock_bitmap(self):
        return get_bitmap(self.view_properties.get_locked_icon())

    def _get_fuzzy_bitmap(self):
        return get_bitmap(self.view_properties.get_fuzzy_icon())


def center_point_with_offset(rect, dx=0, dy=0):
    y = rect.Y + rect.Height // 2 - dy
    x = rect.X + rect.Width // 2 - dx
    return wx.Point(x, y)


def draw_centered_text(dc, rect, label):
    size = dc.GetTextExtent(label)
    point = center_point_with_offset(rect, size.width // 2, size.height // 2)
    dc.DrawText(label, point)


def get_bitmap(name):
    if name not in BITMAPS:
        path = os.path.join(EVENT_ICONS_DIR, name)
        BITMAPS[name] = load_bitmap(path)
    return BITMAPS[name]
