# Copyright (C) 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2018  Rickard Lindberg, Roger Lindberg
#
# This file is part of Timeline.
#
# Timeline is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Timeline is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Timeline.  If not, see <http://www.gnu.org/licenses/>.


from timelinelib.calendar.timetype import TimeType
from timelinelib.calendar.pharaonic.pharaonicdatetime import PharaonicDateTime
from timelinelib.calendar.pharaonic.time import PharaonicDelta
from timelinelib.calendar.pharaonic.time import PharaonicTime
from timelinelib.canvas.data import TimePeriod
from timelinelib.calendar.pharaonic.timetype.strips.stripcentury import StripCentury
from timelinelib.calendar.pharaonic.timetype.strips.stripweek import StripWeek
from timelinelib.calendar.pharaonic.timetype.strips.stripweekday import StripWeekday
from timelinelib.calendar.pharaonic.timetype.strips.stripminute import StripMinute
from timelinelib.calendar.pharaonic.timetype.strips.stripday import StripDay
from timelinelib.calendar.pharaonic.timetype.strips.stripdecade import StripDecade
from timelinelib.calendar.pharaonic.timetype.strips.striphour import StripHour
from timelinelib.calendar.pharaonic.timetype.strips.stripmonth import StripMonth
from timelinelib.calendar.pharaonic.timetype.strips.stripyear import StripYear


class PharaonicTimeType(TimeType):

    @property
    def name(self):
        return "pharaonic"

    @property
    def TimeClass(self):
        return PharaonicTime

    @property
    def DateTimeClass(self):
        return PharaonicDateTime

    @property
    def DeltaClass(self):
        return PharaonicDelta

    def get_navigation_functions(self):
        return [
            (_("Go to &Today") + "\tCtrl+T", self.go_to_today_fn),
            (_("Go to &Date...") + "\tCtrl+G", self.go_to_date_fn),
            ("SEP", None),
            (_("Backward") + "\tPgUp", self.backward_fn),
            (_("Forward") + "\tPgDn", self.forward_fn),
            (_("Forward One Wee&k") + "\tCtrl+K", self.forward_one_week_fn),
            (_("Back One &Week") + "\tCtrl+W", self.backward_one_week_fn),
            (_("Forward One Mont&h") + "\tCtrl+H", self.forward_one_month_fn),
            (_("Back One &Month") + "\tCtrl+M", self.backward_one_month_fn),
            (_("Forward One Yea&r") + "\tCtrl+R", self.forward_one_year_fn),
            (_("Back One &Year") + "\tCtrl+Y", self.backward_one_year_fn),
            ("SEP", None),
            (_("Fit Millennium"), self.fit_millennium_fn),
            (_("Fit Century"), self.create_strip_fitter(StripCentury)),
            (_("Fit Decade"), self.create_strip_fitter(StripDecade)),
            (_("Fit Year"), self.create_strip_fitter(StripYear)),
            (_("Fit Month"), self.create_strip_fitter(StripMonth)),
            (_("Fit Week"), self.fit_week_fn),
            (_("Fit Day"), self.create_strip_fitter(StripDay)),
        ]

    def choose_strip(self, metrics, appearance):
        """
        Return a tuple (major_strip, minor_strip) for current time period and
        window size.
        """
        start_month = PharaonicDateTime.from_time(metrics.time_period.start_time).month
        end_month = PharaonicDateTime.from_time(metrics.time_period.end_time - PharaonicDelta.from_days(1)).month
        five_day_month = start_month == end_month == 13
        day_period = TimePeriod(PharaonicTime(0, 0), PharaonicTime(1, 0))
        one_day_width = metrics.calc_exact_width(day_period)
        if one_day_width > 20000:
            return StripHour(), StripMinute()
        elif one_day_width > 600:
            return StripDay(appearance), StripHour()
        elif one_day_width > 45 and not five_day_month:
            return StripWeek(appearance), StripWeekday()
        elif one_day_width > 25 or five_day_month:
            return StripMonth(), StripDay(appearance)
        elif one_day_width > 1.5:
            return StripYear(), StripMonth()
        elif one_day_width > 0.12:
            return StripDecade(), StripYear()
        elif one_day_width > 0.012:
            return StripCentury(), StripDecade()
        else:
            return StripCentury(), StripCentury()

    def create_time_picker(self, parent, *args, **kwargs):
        from timelinelib.calendar.pharaonic.timepicker.datetimepicker import PharaonicDateTimePicker
        return PharaonicDateTimePicker(parent, *args, **kwargs)
